/*==========================================================================
* DETECT.C - Copyright (c) 1994 ATI Technologies Inc. All rights reserved  *
*                                                                          *
* PGL detection function for Mach64 and I/O base address setup             *
* ======================================================================== */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <conio.h>
#include <memory.h>
#include <dos.h>

#include "..\inc\atim64.h"
#include "..\inc\pgl.h"
#include "..\inc\pglglob.h"

/* --------------------------------------------------------------------------
  PGL_detectmach64 - determine if a mach64 based video adapter is installed.
  
  This routine identifies if a Mach64 based card is installed. The 'instance'
  input parameter is provided to allow selection of a specific Mach64 card in
  systems that permit multiple cards. Normally, set this value to zero. The
  detection is done using the following steps:

    1. Find an ATI Mach64 ROM and its ROM segment by scanning through ROM
       segments C000h-DF00h. To match, the ROM ID, ATI signature, and MACH64
       string must be found.

    2. Call the ROM to find the base I/O address and type (float/standard).

    3. Perform a write/read test on SCRATCH_REG1 (its contents must be saved
       and restored since the ROM functions uses its contents).

    4. Identification of the Mach64 variant is done by obtaining the chip ID
       and version. The base I/O address and type are required first to get
       this information.

  Since this function is the entry point for all PGL functions, it must be
  called first. NO_ERROR is returned if successful, otherwise YES_ERROR is
  returned.
-------------------------------------------------------------------------- */
int PGL_detectmach64(int instance)
{
    PGL_iobasequery iobaseinfo;
    unsigned long save_value;
    int retcode;
    int result;
    int matches;
    int index;
    int exception_flag;
    unsigned int rom_segment[25] =
    {
        0xc000, 0xc800, 0xc900, 0xca00, 0xcb00, 0xcc00, 0xcd00, 0xce00,
        0xcf00, 0xd000, 0xd100, 0xd200, 0xd300, 0xd400, 0xd500, 0xd600,
        0xd700, 0xd800, 0xd900, 0xda00, 0xdb00, 0xdc00, 0xdd00, 0xde00,
        0xdf00
    };

    // assume failure
    result = YES_ERROR;
    exception_flag = 0;
    PGL_modecfg.rom_segment = 0;
    PGL_modecfg.io_base_address = 0;
    PGL_modecfg.io_base_type = IO_TYPE_STANDARD;
    PGL_modecfg.instance = 0;
    PGL_modecfg.chip_type = 0;
    PGL_modecfg.chip_version = 0;
    PGL_modecfg.chip_attribute = CHIP_SP_STANDARD;

    //
    // Step 1 - Find ATI Mach64 ROM
    //

    // attempt to find a matching ROM segment with the correct instance
    matches = 0;
    index = 0;
    while (index < 25)
    {
        // check if ROM belongs to Mach64 product
        if (pgl_isatimach64rom(rom_segment[index]) == 1)
        {
            if (matches == instance)
            {
                // Mach64 ROM was properly identified
                PGL_modecfg.rom_segment = rom_segment[index];
                PGL_modecfg.instance = instance;
                index = 25;
            }
            matches++;
        }
        index++;
    }
    if (PGL_modecfg.rom_segment == 0)
    {
        // return error if no ROM segment could be found
        return (YES_ERROR);
    }

    // setup ROM functions with the correct rom segment
    pgl_setrombase(PGL_modecfg.rom_segment);

    //
    // Step 2 - Get the I/O base address and type
    //

    // get I/O base information from ROM - skip if exception flag is set
    iobaseinfo.io_base = 0;
    iobaseinfo.io_type = 0;

    // call ROM since ATI Mach64 ID was correctly identified
    retcode = ROM_getiobase(&iobaseinfo);
    if ((retcode == NO_ERROR) && (iobaseinfo.io_base != 0))
    {
        // assign valid I/O base address and type from ROM
        PGL_modecfg.io_base_address = iobaseinfo.io_base;
        PGL_modecfg.io_base_type = iobaseinfo.io_type & 1;
    }
    else
    {
        // use default base I/O address in this case
        PGL_modecfg.io_base_address = 0x2ec;
        PGL_modecfg.io_base_type = 0;
    }

    // setup I/O functions with the correct base I/O address and type
    pgl_initioreg(PGL_modecfg.io_base_address, PGL_modecfg.io_base_type);

    //
    // Step 3 - Test a Mach64 I/O register. Note that an ATI Mach32 VLB card
    //          must not be allowed to reach this stage since it does not
    //          support 32 bit I/O writes.
    //

    // save old value
    save_value = ior32(ioSCRATCH_REG1);

    // test even bits for readability
    iow32(ioSCRATCH_REG1, 0x55555555);
    if (ior32(ioSCRATCH_REG1) == 0x55555555)
    {
        // test odd bits for readability
        iow32(ioSCRATCH_REG1, 0xaaaaaaaa);
        if (ior32(ioSCRATCH_REG1) == 0xaaaaaaaa)
        {
            result = NO_ERROR;
        }
    }

    // restore old value
    iow32(ioSCRATCH_REG1, save_value);

    //
    // Step 4 - Identify the Mach64 variant
    //

    // if Mach64 card, initialize some PGL variables
    if (result == NO_ERROR)
    {
        // get chip type and version
        PGL_modecfg.chip_type = (int)(ior16(ioCONFIG_CHIP_ID));
        PGL_modecfg.chip_version = (int)(ior8(ioCONFIG_CHIP_ID + 3));

        // identify chips that require special handling
        if ((PGL_modecfg.chip_type == CHIP_GX_ID) &&
            (PGL_modecfg.chip_version == 0x02) &&
            (PGL_modecfg.io_base_type == IO_TYPE_FLOAT))
        {
            PGL_modecfg.chip_attribute = CHIP_SP_GXE_FLOAT;
        }

        PGL_palSave = 0;
        PGL_modeSet = 0;
        PGL_savecurflag = 0;

        // save text mode for restoration when PGL_closemode() is called
        pgl_savetextmode();

        // set fifo error control
        PGL_setfifocontrol(FIFO_CHECK_NORMAL, FIFO_ACTION_NONE);
    }

    return (result);
}

/* --------------------------------------------------------------------------
  PGL_getversion - returns version of Mach64 PGL.

  This function returns the version of the Mach64 PGL. It can be used to
  determine if certain features are available in earlier versions.
-------------------------------------------------------------------------- */
PGL_version PGL_getversion(void)
{
    PGL_version version;

    version.major = 3;
    version.minor = 34;
    return (version);
}

