/*==========================================================================
  SBUFI.C

  This example code shows how to implement single buffered drawing using
  interrupts. Several image "frames" are loaded into off-screen memory so
  that they can blitted to the screen in sequence. To prevent image tearing,
  a blit is not performed until the display reaches a specific vertical line
  (usually after the lower part of the blit). In this example, this is done
  by setting the VLINE and INT_CNTL registers to trigger an interrupt service
  routine (ISR). For the ISR to function, the Mach64's interrupt line must be
  enabled. On ISA and VLB cards, this program expects IRQ2 to be enabled.
  IRQ2 may be enabled by inserting a jumper on the IRQ2 jumper pins.

  On Mach64 PCI cards, the program queries the interrupt pin field (byte
  offset 3Dh) in the PCI configuration space for the Mach64 card to determine
  if the interrupt line has been enabled. If this value is zero, the
  interrupt is considered to be disabled. Otherwise, the interrupt is assumed
  to be enabled and the interrupt line field (offset 3Ch) is considered to
  contain a valid interrupt number. The program attempts to locate the
  configuration space for all current Mach64 PCI devices until it finds the
  right one (or fails). A table of possible device IDs has been supplied
  for this purpose. An interrupt jumper may also be used to enable the
  interrupt line on PCI Mach64 cards.

  The ISR is chained into the current ISR for the selected interrupt. It
  may fail to function correctly if the IRQ channel is used by other
  hardware or is prevented from functioning.

  Note that the blit code is in the ISR instead of the main loop (as is the
  case for the polled example - SBUFP).

  The frame rate is determined by the vertical frequency of the display.

  Copyright (c) 1994-1995 ATI Technologies Inc. All rights reserved
 =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>
#include "..\util\atim64.h"
#include "..\util\sample.h"
#include "..\util\vint.h"
#include "..\util\vtga.h"
#include "..\util\pci.h"
#include "visr.h"

#define INSTANCE 0

#define IRQ                 2
#define PCI_IRQ_LINE        0x3C
#define PCI_IRQ_PIN         0x3D
#define IRQ_TIMEOUT         30
#define BACKGROUND_COLOR    0x11


/* Main C program */

int main (void)
{
    char filename[20];
    TARGA_HEADER header;
    int irq_level;
    int irq_pin;
    int width, height;
    int srcx, srcy;
    int savex, savey;
    int i, j;
    int y_draw;
    int update;
    int frame;
    int frames;
    int framesperwidth;
    int topline;
    int step;
    int oldcount;
    unsigned int starttick, endtick;
    unsigned long save_bus_cntl;
    POINT points[8];
    PCI_INFO pci_info;
    PCI_DEVICE_INFO pci_device_info;
    unsigned pci_device_ID[] = {0x4354, 0x4758, 0x4358}; // known device IDs

    // check if Mach64 adapter is installed
    if (detect_mach64 (INSTANCE) != YES_MACH64)
    {
        printf ("mach64 based adapter was not found.\n");
        return (1);
    }

    // fill global query structure by calling Mach 64 ROM
    if (query_hardware () != NO_ERROR)
    {
        printf ("Failed ROM call to query mach64 hardware.\n");
        return (1);
    }

    // check if Mach 64 VGA controller is enabled
    if (querydata.vga_type != VGA_ENABLE)
    {
        printf ("This sample code example requires an enabled mach64 VGA "
                "controller.\n");
        return (1);
    }

    // set an accelerator mode
    if (open_mode (MODE_640x480, PITCH_XRES, COLOR_DEPTH_8) != NO_ERROR)
    {
        printf ("Error in setting display mode.\n");
        return (1);
    }

    // check if color depth is 8 bpp
    if (modeinfo.bpp != 8)
    {
        close_mode ();
        printf ("This example requires an 8 bpp mode.\n");
        return (1);
    }

    // determine IRQ level to use for ISR
    if (querydata.bus_type == BUS_PCI)
    {
        if (!pci_bios_present (&pci_info))
        {
            close_mode ();
            printf ("A PCI BIOS has not been detected.\n");
            return (1);
        }

        if (pci_info.present_status != 0)
        {
            close_mode ();
            printf ("The IFF EDX has not been set properly.\n");
            return (1);
        }

        // get the PCI number for the current device
        for (i = 0; i < sizeof (pci_device_ID) / sizeof (unsigned); i++)
        {
            if (pci_find_device (pci_device_ID[i], VENDOR_ID, 0,
                                 &pci_device_info) == SUCCESS) break;
        }

        if (i == sizeof (pci_device_ID) / sizeof (unsigned))
        {
            close_mode ();
            printf ("Could not get PCI device number.\n");
            return (1);
        }

        // determine if the interrupt line has been enabled
        if ((irq_pin = pci_read_config_byte (PCI_IRQ_PIN, &pci_device_info)) ==
            BAD_REGISTER_NUMBER)
        {
            close_mode ();
            printf ("Error reading interrupt pin number.\n");
            return (1);
        }

        if (irq_pin == 0)
        {
            close_mode ();
            printf ("This graphics card does not contain an interrupt line.\n");
            return (1);
        }

        // get the interrupt line level
        irq_level = pci_read_config_byte (PCI_IRQ_LINE, &pci_device_info);
        if (irq_level == BAD_REGISTER_NUMBER)
        {
            close_mode ();
            printf ("Error reading interrupt line level.\n");
            return (1);
        }
    }
    else
    {
        irq_level = IRQ;
    }

    // setup engine context and clear screen
    init_engine ();
    clear_screen (0, 0, modeinfo.xres, modeinfo.yres);

    // get targa header information
    if (get_targa_header ("..\\image\\frame1.tga", &header) != SUCCESS)
    {
        close_mode ();
        printf ("Error reading targa file header information.\n");
        return (1);
    }

    // setup image size, source area, save area, and position increment
    width = header.width & 0xFFFE;
    height = header.height & 0xFFFE;
    srcx = 0;
    srcy = modeinfo.yres + height;
    savex = 0;
    savey = modeinfo.yres;
    y_draw = modeinfo.yres - height;
    step = 2;

    // setup blit dimensions for ISR
    setblitinfo (width, height, srcx, srcy, savex, savey, y_draw, step);
    setxstart (0);
    setxpos (0);
    setimage (0, srcx, srcy);

    // determine how large to expand the scissors
    frames = 8;
    framesperwidth = modeinfo.xres / width;
    topline = frames / framesperwidth;
    if ((topline * framesperwidth) != frames)
    {
        topline++;
    }
    topline = ((topline + 1) * height) + modeinfo.yres;

    // expand scissors to include source and save areas
    wait_for_fifo (4);
    regw (SC_LEFT, 0);
    regw (SC_TOP, 0);
    regw (SC_RIGHT, modeinfo.xres - 1);
    regw (SC_BOTTOM, topline - 1);

    // set background color the same as the image background
    set_fg_color (BACKGROUND_COLOR);
    draw_rectangle (0, 0, modeinfo.xres, modeinfo.yres);

    // load source images into off-screen memory for frame blitting and
    // record their position
    frame = 0;
    i = 0;
    j = 0;
    while (frame < frames)
    {
        // record each frame coordinate
        points[frame].x = srcx + (width * i);
        points[frame].y = srcy + (height * j);

        // load next frame image into video memory
        sprintf (filename, "..\\image\\frame%d.tga", frame + 1);
        if (load_targa (filename, points[frame].x, points[frame].y) != SUCCESS)
        {
            close_mode ();
            printf ("Error loading targa file to memory.\n");
            return (1);
        }

        // adjust location of frame load coordinate as necessary
        frame++;
        i++;
        if (i > ((modeinfo.xres / header.width) - 1))
        {
            i = 0;
            j++;
        }
    }

    // set palette from targa color table (8 bpp)
    if (header.pixel_depth == 8)
    {
        if (set_targa_palette ("..\\image\\frame1.tga") != SUCCESS)
        {
            close_mode ();
            printf ("Error reading targa file color table information.\n");
            return (1);
        }
    }

    // wait for a key to start
    getch ();

    // setup engine for blits
    wait_for_fifo (3);
    regw (DP_SRC, FRGD_SRC_BLIT);
    regw (SRC_CNTL, 0);
    regw (DST_CNTL, DST_Y_TOP_TO_BOTTOM | DST_X_LEFT_TO_RIGHT);

    // Set bus wait to maximum to insure that interrupt clearing occurs
    save_bus_cntl = ior32 (ioBUS_CNTL);
    iow32 (ioBUS_CNTL, save_bus_cntl | 0x0F);

    // set vline to wait for (to prevent image tearing) and enable
    if (y_draw < (modeinfo.yres - height))
    {
        set_vline (y_draw + height);
    }
    else
    {
        set_vline (modeinfo.yres);
    }
    enable_vlineint ();

    // enable ISR and start draw loop
    initvlineisr (irq_level);
    update = getcount ();
    oldcount = update;
    i = 0;
    frame = 0;
    while (i < (modeinfo.xres - width))
    {
        // syncronize with ISR (vline trigger) - time out if no response
        starttick = *((unsigned int far *) (DOS_TICK_ADDRESS));
        endtick = starttick;
        while (update == oldcount)
        {
            update = getcount ();
            endtick = *((unsigned int far *) (DOS_TICK_ADDRESS));
            if (abs (endtick - starttick) > IRQ_TIMEOUT)
            {
                disable_vlineint ();
                cancelvlineisr ();
                close_mode ();
                printf ("IRQ channel %d is not responding.\n", irq_level);
                return (1);
            }
        }
        oldcount = update;

        // cycle through frames
        setimage (frame, points[frame].x, points[frame].y);
        frame++;
        if (frame >= frames)
        {
            frame = 0;
        }

        // increment position
        i = i + step;
        setxpos (i);
    }

    // disable vline interrupts
    disable_vlineint ();

    cancelvlineisr ();

    // restore bus wait states to original value
    iow32 (ioBUS_CNTL, save_bus_cntl);

    // wait for a carriage return
    getch ();

    // disable accelerator mode and switch back to VGA text mode
    close_mode ();

    return (0);
}

